<?php
// =========================================================================
// definir_prazo_lancamento.php - Interface para a Subdireção Pedagógica
// Define e ativa o prazo limite para o lançamento de notas (Item III).
// =========================================================================
 
// --- CONFIGURAÇÃO DE CONEXÃO ---
define('DB_HOST', 'localhost');
define('DB_PORT', '3307');
define('DB_NAME', 'sismag'); 
define('DB_USER', 'root');
define('DB_PASS', '');
 
$pdo = null;
$mensagem = '';
 
try {
    $dsn = "mysql:host=" . DB_HOST . ";port=" . DB_PORT . ";dbname=" . DB_NAME . ";charset=utf8";
    $pdo = new PDO($dsn, DB_USER, DB_PASS);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $pdo->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die("ERRO DE CONEXÃO COM A BASE DE DADOS 'sismag': " . $e->getMessage());
}

// --- DEFINIÇÕES DE DATA E TRIMESTRE ---
$ANO_ATUAL = date('Y'); 
$ANOS_DISPONIVEIS = range($ANO_ATUAL - 1, $ANO_ATUAL + 2); 
$TRIMESTRES_DISPONIVEIS = [
    1 => 'I Trimestre',
    2 => 'II Trimestre',
    3 => 'III Trimestre',
    4 => 'IV Trimestre',
];
// ID simulado para o responsável (deve ser o ID do usuário logado)
$RESPONSAVEL_ID = 99; 
 
// --- LÓGICA DE SALVAR O PRAZO ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['salvar_prazo'])) {
    
    $ano = (int)$_POST['ano'];
    $trimestre = (int)$_POST['trimestre'];
    $data_limite = $_POST['data_limite']; 

    if (empty($data_limite) || !checkdate(date('m', strtotime($data_limite)), date('d', strtotime($data_limite)), date('Y', strtotime($data_limite)))) {
         $mensagem = "<p style='color:red;'>Erro: Data limite inválida.</p>";
    } elseif (!in_array($ano, $ANOS_DISPONIVEIS) || !isset($TRIMESTRES_DISPONIVEIS[$trimestre])) {
         $mensagem = "<p style='color:red;'>Erro: Ano ou Trimestre inválido.</p>";
    } else {
        try {
            // Tenta inserir o novo prazo, ou atualiza se já existir
            $stmt = $pdo->prepare("
                INSERT INTO prazo_lancamento_notas 
                (ano, trimestre, data_limite, responsavel_id, data_definicao) 
                VALUES (?, ?, ?, ?, NOW())
                ON DUPLICATE KEY UPDATE 
                data_limite = VALUES(data_limite), responsavel_id = VALUES(responsavel_id), data_definicao = NOW()
            ");
            $stmt->execute([
                $ano, 
                $trimestre, 
                $data_limite, 
                $RESPONSAVEL_ID
            ]);

            $mensagem = "<p style='color:green; font-weight:bold;'>Sucesso! Prazo limite para o {$TRIMESTRES_DISPONIVEIS[$trimestre]} / {$ano} foi definido para " . date('d/m/Y', strtotime($data_limite)) . ".</p>";
            
        } catch (PDOException $e) {
            $mensagem = "<p style='color:red;'>Erro ao salvar no banco de dados: " . $e->getMessage() . "</p>";
        }
    }
}

// --- BUSCA PRAZOS ATIVOS PARA EXIBIÇÃO ---
$stmt_prazos = $pdo->prepare("
    SELECT ano, trimestre, data_limite, data_definicao 
    FROM prazo_lancamento_notas 
    WHERE ano >= ?
    ORDER BY ano DESC, trimestre ASC
");
$stmt_prazos->execute([$ANO_ATUAL - 1]);
$prazos_existentes = $stmt_prazos->fetchAll();

?>
<!DOCTYPE html>
<html lang="pt">
<head>
    <meta charset="UTF-8">
    <title>Gestão de Prazo de Lançamento de Notas</title>
    <style>
        /* ... Estilos CSS ... */
        body { font-family: Arial, sans-serif; background-color: #f4f4f4; margin: 20px; }
        .container { max-width: 800px; margin: 0 auto; background-color: #fff; padding: 25px; border-radius: 8px; box-shadow: 0 0 15px rgba(0,0,0,0.1); }
        h1 { text-align: center; color: #dc3545; border-bottom: 3px solid #dc3545; padding-bottom: 10px; margin-bottom: 25px; }
        h2 { color: #007bff; margin-top: 30px; border-bottom: 1px solid #ccc; padding-bottom: 5px; }
        .form-group { margin-bottom: 15px; display: flex; gap: 20px; align-items: flex-end; }
        .form-field { flex: 1; }
        label { display: block; margin-bottom: 5px; font-weight: bold; color: #555; }
        input[type="date"], select { padding: 10px; border: 1px solid #ccc; border-radius: 4px; box-sizing: border-box; width: 100%; }
        button { 
            background-color: #28a745; 
            color: white; 
            padding: 10px 15px; 
            border: none; 
            border-radius: 4px; 
            cursor: pointer; 
            font-size: 1em;
            flex-shrink: 0;
            height: 40px; 
            margin-top: 5px;
        }
        .tabela-prazos { width: 100%; border-collapse: collapse; margin-top: 20px; }
        .tabela-prazos th, .tabela-prazos td { border: 1px solid #ddd; padding: 10px; text-align: left; }
        .tabela-prazos th { background-color: #007bff; color: white; }
        /* ... Fim dos Estilos CSS ... */
    </style>
</head>
<body>
    <div class="container">
        <h1>Gestão de Prazo - Lançamento de Notas</h1>
        <p style="text-align: center; color: #dc3545; font-weight: bold;">
            (Área reservada à Subdireção Pedagógica)
        </p>

        <?= $mensagem ?>

        <form action="" method="post">
            <h2>Definir Novo Prazo Limite</h2>
            <div class="form-group">
                <div class="form-field">
                    <label for="ano">Ano Lectivo:</label>
                    <select id="ano" name="ano" required>
                        <?php foreach ($ANOS_DISPONIVEIS as $ano): ?>
                            <option value="<?= $ano ?>" <?= ($ano == $ANO_ATUAL) ? 'selected' : '' ?>>
                                <?= $ano ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-field">
                    <label for="trimestre">Trimestre:</label>
                    <select id="trimestre" name="trimestre" required>
                        <?php foreach ($TRIMESTRES_DISPONIVEIS as $num => $nome): ?>
                            <option value="<?= $num ?>"><?= $nome ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="form-field">
                    <label for="data_limite">Data Limite Final de Lançamento:</label>
                    <input type="date" id="data_limite" name="data_limite" required>
                </div>
                
                <button type="submit" name="salvar_prazo">Salvar Prazo</button>
            </div>
        </form>

        <h2>Prazos Limites Definidos</h2>
        <?php if (empty($prazos_existentes)): ?>
            <p style="color: gray; text-align: center;">Nenhum prazo limite foi definido ainda.</p>
        <?php else: ?>
            <table class="tabela-prazos">
                <thead>
                    <tr>
                        <th>Ano</th>
                        <th>Trimestre</th>
                        <th>Data Limite</th>
                        <th>Definido em</th>
                        <th>Status</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($prazos_existentes as $prazo): 
                        $data_limite_obj = new DateTime($prazo['data_limite']);
                        $data_hoje = new DateTime(date('Y-m-d'));
                        
                        if ($data_limite_obj < $data_hoje) {
                            $status = '<span style="color: red; font-weight: bold;">EXPIRADO</span>';
                        } elseif ($data_limite_obj == $data_hoje) {
                            $status = '<span style="color: orange; font-weight: bold;">HOJE É O PRAZO</span>';
                        } else {
                            $status = '<span style="color: green;">Ativo</span>';
                        }
                    ?>
                        <tr>
                            <td><?= $prazo['ano'] ?></td>
                            <td><?= $TRIMESTRES_DISPONIVEIS[$prazo['trimestre']] ?></td>
                            <td><?= date('d/m/Y', strtotime($prazo['data_limite'])) ?></td>
                            <td><?= date('d/m/Y H:i', strtotime($prazo['data_definicao'])) ?></td>
                            <td><?= $status ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</body>
</html>

    


